#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.DAL.UnitTests
{

	[TestFixture]
	public class DAL_LocalSupplier : BaseTest 
	{
		private DataTable _dtSupplier;
		private DataTable _dtInactiveSupplier;
		private DataTable _dtActiveProducts;
		private DataTable _dtActiveUnits;
		private DataTable _dtAddress;

		[SetUp]
		protected void SetUp() 
		{				
			this.BaseSetUp();

			if (this.RefreshData)
			{
				//Create some data that's never out there
				DataCreator.CreateBloodUnit(true);

				//Get Inactive supplier  -- Can't use V_SUppliers -- that only returns active
				this._dtInactiveSupplier = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM LocalSupplier WHERE LocalSupplierGuid NOT IN (SELECT LocalSupplierGuid FROM VamcDivision) AND RecordStatusCode = 'X' AND ICCBBARegistrationNumber <> '' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", false);
				if (this._dtInactiveSupplier.Rows.Count == 0)
				{
					BOL.LocalSupplier tmpLS1 = DataCreator.CreateLocalSupplier();
					UnitTestUtil.RunSQL("UPDATE LocalSupplier SET RecordStatusCode = 'X' WHERE LocalSupplierGuid = '" + tmpLS1.LocalSupplierGuid.ToString() + "'");
					this._dtInactiveSupplier = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM LocalSupplier WHERE LocalSupplierGuid = '" + tmpLS1.LocalSupplierGuid.ToString() + "'", true);
				}

				DataCreator.CreateLocalSupplier();
				this._dtSupplier = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM V_Suppliers WHERE LocalSupplierGuid NOT IN (SELECT LocalSupplierGuid FROM VamcDivision) AND AddressGuid IS NOT NULL AND RecordStatusCode = 'L' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' ORDER BY LastUpdateDate DESC", true);

				//Get Address information for our above supplier
				this._dtAddress = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Address WHERE AddressGuid = '" + this._dtSupplier.Rows[0][TABLES.LocalSupplier.AddressGuid].ToString() + "'", true);

				DataCreator.CreateBloodUnit(true);
				//spIsLocalSupplierOnActiveBloodProducts
				this._dtActiveProducts = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM ProductShipper PS INNER JOIN LocalSupplier LS ON PS.LocalSupplierGUID = LS.LocalSupplierGUID WHERE PS.RecordStatusCode != 'X' AND PS.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);

				DataCreator.CreateBloodUnit(true);
				//spIsLocalSupplierOnActiveBloodUnits
				this._dtActiveUnits = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM BloodUnit WHERE UnitFDARegistrationNumber IN (SELECT FdaRegistrationNumber FROM LocalSupplier WHERE IccbbaRegistrationNumber IS NOT NULL AND DivisionCode = '" + this.TestDivisionCode + "') AND RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' ORDER BY LastUpdateDate DESC", true);


				this.RefreshData = false;
			}

		}

		[Test]
		public void GetFacilityDefinitionByGuid_Pass()
		{
			DataRow drTest = DAL.LocalSupplier.GetFacilityDefinitionByGuid((System.Guid)this._dtSupplier.Rows[0][TABLES.Facility.FacilityGuid]);

			Assert.AreEqual(4, drTest.Table.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, drTest.Table.Columns.Contains(TABLES.Facility.FacilityGuid), TABLES.Facility.FacilityGuid);
			Assert.AreEqual(true, drTest.Table.Columns.Contains(TABLES.Facility.FacilityName), TABLES.Facility.FacilityName);
			Assert.AreEqual(true, drTest.Table.Columns.Contains(TABLES.Facility.FdaRegistrationNumber), TABLES.Facility.FdaRegistrationNumber);
			Assert.AreEqual(true, drTest.Table.Columns.Contains(TABLES.Facility.RecordStatusCode), TABLES.Facility.RecordStatusCode);
		}

		[Test]
        [ExpectedException(typeof(Common.DataAccessLayerException))]
		public void GetFacilityDefinitionByGuid_Fail()
		{
			DAL.LocalSupplier.GetFacilityDefinitionByGuid(System.Guid.NewGuid());
		}

		[Test]
		public void GetLocalSupplierLookupList_Pass()
		{
			DataTable dtTest = DAL.LocalSupplier.GetLocalSupplierLookupList(this._dtSupplier.Rows[0][TABLES.LocalSupplier.FacilityName].ToString(), this._dtSupplier.Rows[0][TABLES.LocalSupplier.FdaRegistrationNumber].ToString());

			VerifyLookupListResultDataTable( dtTest );
		}

		[Test]
		public void GetLocalSupplierLookupList_Fail()
		{
			DataTable dtTest = DAL.LocalSupplier.GetLocalSupplierLookupList("Greg", "Lohse");

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetNationalSupplierLookupList_Pass()
		{
			DataTable dtTest = DAL.LocalSupplier.GetFacilityNationalLookupList();

			Assert.AreEqual(3, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Facility.FacilityGuid), TABLES.Facility.FacilityGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FdaRegistrationNumber), TABLES.LocalSupplier.FdaRegistrationNumber);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetNationalSupplierLookupList_Fail()
		{
		}		

		[Test]
		public void GetLocalSuppliersForDivision_Pass()
		{
			DataTable dtTest = DAL.LocalSupplier.GetLocalSuppliersForDivision();

			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
			Assert.AreEqual(2, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.LocalSupplierGuid), TABLES.LocalSupplier.LocalSupplierGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetLocalSuppliersForDivision_Fail()
		{
		}

		[Test]
		public void GetLocalSupplierDetailByFdaRegNum_Pass()
		{
			DataTable dtTestActiveOnly = DAL.LocalSupplier.GetLocalSupplierDetailByFdaRegNum(this._dtInactiveSupplier.Rows[0][TABLES.LocalSupplier.FdaRegistrationNumber].ToString(), true);
			DataTable dtTestInactive = DAL.LocalSupplier.GetLocalSupplierDetailByFdaRegNum(this._dtInactiveSupplier.Rows[0][TABLES.LocalSupplier.FdaRegistrationNumber].ToString(), false);
			
			Assert.AreEqual(0, dtTestActiveOnly.Rows.Count, "ActiveOnly Rows.Count");
			Assert.AreEqual(true, dtTestInactive.Rows.Count > 0, "Inactive Rows.Count");

			Assert.AreEqual(22, dtTestActiveOnly.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.LocalSupplierGuid), TABLES.LocalSupplier.LocalSupplierGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityGuid), TABLES.LocalSupplier.FacilityGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FdaRegistrationNumber), TABLES.LocalSupplier.FdaRegistrationNumber);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.IccbbaRegistrationNumber), TABLES.LocalSupplier.IccbbaRegistrationNumber);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityPrefix), TABLES.LocalSupplier.FacilityPrefix);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.AddressGuid), TABLES.LocalSupplier.AddressGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.LabelUnitIdTrimIndicator), TABLES.LocalSupplier.LabelUnitIdTrimIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.PhoneNumber), TABLES.LocalSupplier.PhoneNumber);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FaxNumber), TABLES.LocalSupplier.FaxNumber);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.TestingIndicator), TABLES.LocalSupplier.TestingIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.CollectionIndicator), TABLES.LocalSupplier.CollectionIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.DivisionCode), TABLES.LocalSupplier.DivisionCode);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.RecordStatusCode), TABLES.LocalSupplier.RecordStatusCode);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.RowVersion), TABLES.LocalSupplier.RowVersion);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine1), TABLES.Address.AddressLine1);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine2), TABLES.Address.AddressLine2);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine3), TABLES.Address.AddressLine3);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressCity), TABLES.Address.AddressCity);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressZipCode), TABLES.Address.AddressZipCode);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.StateCode), TABLES.Address.StateCode);
			Assert.AreEqual(true , dtTestActiveOnly.Columns.Contains(ARTIFICIAL.AddressRowVersion), ARTIFICIAL.AddressRowVersion);
		}

		[Test]
		public void GetLocalSupplierDetailByFdaRegNum_Fail()
		{
			DataTable dtTest = DAL.LocalSupplier.GetLocalSupplierDetailByFdaRegNum(string.Empty, true);
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}


		[Test]
		public void GetLocalSupplierDetailByIccbaRegNum_Pass()
		{
			DataTable dtTestActiveOnly = DAL.LocalSupplier.GetLocalSupplierDetailByIccbaRegNum(this._dtInactiveSupplier.Rows[0][TABLES.LocalSupplier.IccbbaRegistrationNumber].ToString(), true);
			DataTable dtTestInactive = DAL.LocalSupplier.GetLocalSupplierDetailByIccbaRegNum(this._dtInactiveSupplier.Rows[0][TABLES.LocalSupplier.IccbbaRegistrationNumber].ToString(), false);
			
			Assert.AreEqual(true, dtTestActiveOnly.Rows.Count <= 1, "ActiveOnly Rows.Count");	//National will be brought back
			Assert.AreEqual(true, dtTestInactive.Rows.Count > 0, "Inactive Rows.Count");

			Assert.AreEqual(22, dtTestActiveOnly.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.LocalSupplierGuid), TABLES.LocalSupplier.LocalSupplierGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityGuid), TABLES.LocalSupplier.FacilityGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FdaRegistrationNumber), TABLES.LocalSupplier.FdaRegistrationNumber);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.IccbbaRegistrationNumber), TABLES.LocalSupplier.IccbbaRegistrationNumber);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityPrefix), TABLES.LocalSupplier.FacilityPrefix);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.AddressGuid), TABLES.LocalSupplier.AddressGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.LabelUnitIdTrimIndicator), TABLES.LocalSupplier.LabelUnitIdTrimIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.PhoneNumber), TABLES.LocalSupplier.PhoneNumber);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FaxNumber), TABLES.LocalSupplier.FaxNumber);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.TestingIndicator), TABLES.LocalSupplier.TestingIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.CollectionIndicator), TABLES.LocalSupplier.CollectionIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.DivisionCode), TABLES.LocalSupplier.DivisionCode);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.RecordStatusCode), TABLES.LocalSupplier.RecordStatusCode);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.RowVersion), TABLES.LocalSupplier.RowVersion);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine1), TABLES.Address.AddressLine1);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine2), TABLES.Address.AddressLine2);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine3), TABLES.Address.AddressLine3);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressCity), TABLES.Address.AddressCity);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressZipCode), TABLES.Address.AddressZipCode);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.StateCode), TABLES.Address.StateCode);
			Assert.AreEqual(true , dtTestActiveOnly.Columns.Contains(ARTIFICIAL.AddressRowVersion), ARTIFICIAL.AddressRowVersion);
		}

		[Test]
		public void GetLocalSupplierDetailByIccbaRegNum_Fail()
		{
			DataTable dtTest = DAL.LocalSupplier.GetLocalSupplierDetailByIccbaRegNum("@XXX@", true);
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetLocalSupplierDetailByGuid_Pass()
		{
			DataTable dtTestActiveOnly = DAL.LocalSupplier.GetLocalSupplierDetailByGuid((System.Guid)this._dtInactiveSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid], true);
			DataTable dtTestInactive = DAL.LocalSupplier.GetLocalSupplierDetailByGuid((System.Guid)this._dtInactiveSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid], false);

			Assert.AreEqual(0, dtTestActiveOnly.Rows.Count, "ActiveOnly Rows.Count");
			Assert.AreEqual(true, dtTestInactive.Rows.Count > 0, "Inactive Rows.Count");
			Assert.AreEqual(22, dtTestActiveOnly.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.LocalSupplierGuid), TABLES.LocalSupplier.LocalSupplierGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityGuid), TABLES.LocalSupplier.FacilityGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FdaRegistrationNumber), TABLES.LocalSupplier.FdaRegistrationNumber);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.IccbbaRegistrationNumber), TABLES.LocalSupplier.IccbbaRegistrationNumber);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FacilityPrefix), TABLES.LocalSupplier.FacilityPrefix);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.AddressGuid), TABLES.LocalSupplier.AddressGuid);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.LabelUnitIdTrimIndicator), TABLES.LocalSupplier.LabelUnitIdTrimIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.PhoneNumber), TABLES.LocalSupplier.PhoneNumber);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.FaxNumber), TABLES.LocalSupplier.FaxNumber);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.TestingIndicator), TABLES.LocalSupplier.TestingIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.CollectionIndicator), TABLES.LocalSupplier.CollectionIndicator);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.DivisionCode), TABLES.LocalSupplier.DivisionCode);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.RecordStatusCode), TABLES.LocalSupplier.RecordStatusCode);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.LocalSupplier.RowVersion), TABLES.LocalSupplier.RowVersion);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine1), TABLES.Address.AddressLine1);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine2), TABLES.Address.AddressLine2);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressLine3), TABLES.Address.AddressLine3);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressCity), TABLES.Address.AddressCity);
			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.AddressZipCode), TABLES.Address.AddressZipCode);

			Assert.AreEqual(true, dtTestActiveOnly.Columns.Contains(TABLES.Address.StateCode), TABLES.Address.StateCode);
			Assert.AreEqual(true , dtTestActiveOnly.Columns.Contains(ARTIFICIAL.AddressRowVersion), ARTIFICIAL.AddressRowVersion);
		}

		[Test]
		public void GetLocalSupplierDetailByGuid_Fail()
		{
			DataTable dtTest = DAL.LocalSupplier.GetLocalSupplierDetailByGuid(System.Guid.NewGuid(), true);
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void IsValidFdaRegistrationNumber_Pass()
		{
			bool retValid = DAL.LocalSupplier.IsValidFdaRegistrationNumber(this._dtSupplier.Rows[0][TABLES.LocalSupplier.FdaRegistrationNumber].ToString());
			Assert.AreEqual(true, retValid, "ReturnValue");
		}

		[Test]
		public void IsValidFdaRegistrationNumber_Fail()
		{
			bool retInvalid = DAL.LocalSupplier.IsValidFdaRegistrationNumber(System.DateTime.Now.Year.ToString());
			Assert.AreEqual(false, retInvalid, "ReturnValue");
		}

		[Test]
		public void IsLocalSupplierOnActiveBloodProducts_Pass()
		{
			bool retValue = DAL.LocalSupplier.IsLocalSupplierOnActiveBloodProducts((System.Guid)this._dtActiveProducts.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid]);
			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		public void IsLocalSupplierOnActiveBloodProducts_Fail()
		{
			bool retValue = DAL.LocalSupplier.IsLocalSupplierOnActiveBloodProducts(System.Guid.NewGuid());
			Assert.AreEqual(false, retValue, "ReturnValue");

		}

		[Test]
		public void IsLocalSupplierActiveByIccbbaRegistrationNumber_Pass()
		{
			bool retValue = DAL.LocalSupplier.IsLocalSupplierActiveByIccbbaRegistrationNumber(this._dtSupplier.Rows[0][TABLES.LocalSupplier.IccbbaRegistrationNumber].ToString());
			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		public void IsLocalSupplierActiveByIccbbaRegistrationNumber_Fail()
		{
			bool retValue = DAL.LocalSupplier.IsLocalSupplierActiveByIccbbaRegistrationNumber("HOMER");
			Assert.AreEqual(false, retValue, "ReturnValue");
		}

		[Test]
		public void IsLocalSupplierOnActiveBloodUnits_Pass()
		{
			bool retValue = DAL.LocalSupplier.IsLocalSupplierOnActiveBloodUnits(this._dtActiveUnits.Rows[0][TABLES.BloodUnit.UnitFdaRegistrationNumber].ToString());
			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		public void IsLocalSupplierOnActiveBloodUnits_Fail()
		{
			bool retValue = DAL.LocalSupplier.IsLocalSupplierOnActiveBloodUnits("HOMER12");
			Assert.AreEqual(false, retValue, "ReturnValue");
		}

		[Test]
		public void InsertLocalSupplier_Pass()
		{
			this.RefreshData = true;

			DataTable dtInsertLS = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(false);

			//Create the new LocalSupplier Entry
			DataRow drInsLS = dtInsertLS.NewRow();
			drInsLS[TABLES.LocalSupplier.LocalSupplierGuid] = System.Guid.NewGuid();
			drInsLS[TABLES.LocalSupplier.FacilityName] = "Blood Bath and Beyond";
			drInsLS[TABLES.LocalSupplier.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.LocalEntry);
			drInsLS[TABLES.LocalSupplier.FacilityGuid] = this._dtSupplier.Rows[0][TABLES.LocalSupplier.FacilityGuid];
			drInsLS[TABLES.LocalSupplier.AddressGuid] = this._dtAddress.Rows[0][TABLES.Address.AddressGuid];
			drInsLS[TABLES.LocalSupplier.FdaRegistrationNumber] = new Random().Next(11111,99999).ToString() + new Random().Next(11111,99999).ToString();
			drInsLS[TABLES.LocalSupplier.DivisionCode] = this.TestDivisionCode;

			DataTable dtUpdateAddress = DAL.Address.GetEmptyAddressTableSchema(true);

			DataRow drUpdAdd = dtUpdateAddress.NewRow();
			drUpdAdd[TABLES.Address.AddressGuid] = this._dtAddress.Rows[0][TABLES.Address.AddressGuid];
			drUpdAdd[TABLES.Address.AddressCity] = "West Chicago";
			drUpdAdd[TABLES.Address.AddressZipCode] = "60185";
			drUpdAdd[TABLES.Address.StateCode] = "IL";
			drUpdAdd[ARTIFICIAL.AddressRowVersion] = this._dtAddress.Rows[0][TABLES.Address.RowVersion];
			dtUpdateAddress.Rows.Add(drUpdAdd);

			Common.Utility.AppendLastUpdateInformation(dtInsertLS, Common.UpdateFunction.UnitTests);
			Common.Utility.AppendLastUpdateInformation(dtUpdateAddress, Common.UpdateFunction.UnitTests);

			bool retValue = DAL.LocalSupplier.InsertLocalSupplier(dtInsertLS, dtUpdateAddress, false);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(Common.UniqueDatabaseConstraintViolatedException))]
		public void InsertLocalSupplier_Fail()
		{
			this.RefreshData = true;

			DataTable dtInsertLS = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(false);

			//Create the new LocalSupplier Entry
			DataRow drInsLS = dtInsertLS.NewRow();
			drInsLS[TABLES.LocalSupplier.LocalSupplierGuid] = this._dtSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid];
			drInsLS[TABLES.LocalSupplier.FacilityName] = "Blood Bath and Beyond";
			drInsLS[TABLES.LocalSupplier.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.LocalEntry);
			drInsLS[TABLES.LocalSupplier.FacilityGuid] = this._dtSupplier.Rows[0][TABLES.LocalSupplier.FacilityGuid];
			drInsLS[TABLES.LocalSupplier.AddressGuid] = this._dtAddress.Rows[0][TABLES.Address.AddressGuid];
			drInsLS[TABLES.LocalSupplier.FdaRegistrationNumber] = new Random().Next(11111,99999).ToString() + new Random().Next(11111,99999).ToString();
			drInsLS[TABLES.LocalSupplier.DivisionCode] = this.TestDivisionCode;
			dtInsertLS.Rows.Add(drInsLS);

			DataTable dtUpdateAddress = DAL.Address.GetEmptyAddressTableSchema(false);

			DataRow drUpdAdd = dtUpdateAddress.NewRow();
			drUpdAdd[TABLES.Address.AddressGuid] = System.Guid.NewGuid();
			drUpdAdd[TABLES.Address.AddressCity] = "West Chicago";
			drUpdAdd[TABLES.Address.AddressZipCode] = "60185";
			drUpdAdd[TABLES.Address.StateCode] = "IL";
			dtUpdateAddress.Rows.Add(drUpdAdd);

			Common.Utility.AppendLastUpdateInformation(dtInsertLS, Common.UpdateFunction.UnitTests);
			Common.Utility.AppendLastUpdateInformation(dtUpdateAddress, Common.UpdateFunction.UnitTests);

			DAL.LocalSupplier.InsertLocalSupplier(dtInsertLS, dtUpdateAddress, true);

		}

		[Test]
		public void UpdateLocalSupplier_Pass()
		{
			this.RefreshData = true;

			DataTable dtDataLS = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM LocalSupplier WHERE LocalSupplierGuid = '" + this._dtSupplier.Rows[0]["LocalSupplierGuid"].ToString() + "'");
			DataTable dtDataAD = UnitTestUtil.RunSQLGetDataTable("SELECT * FROM Address WHERE AddressGuid = '" + dtDataLS.Rows[0][TABLES.LocalSupplier.AddressGuid].ToString() + "'");

			DataTable dtUpdateLS = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(true);

			DataRow drUpdLS = dtUpdateLS.NewRow();
			drUpdLS[TABLES.LocalSupplier.LocalSupplierGuid] = dtDataLS.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid];
			drUpdLS[TABLES.LocalSupplier.FacilityName] = "UnitTestChanged-" + System.DateTime.Now.ToString();
			drUpdLS[TABLES.LocalSupplier.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.LocalEntry);
			drUpdLS[TABLES.LocalSupplier.RowVersion] = dtDataLS.Rows[0][TABLES.LocalSupplier.RowVersion];
			dtUpdateLS.Rows.Add(drUpdLS);

			DataTable dtUpdateAddress = DAL.Address.GetEmptyAddressTableSchema(true);

			DataRow drUpdAdd = dtUpdateAddress.NewRow();
			drUpdAdd[TABLES.Address.AddressGuid] = dtDataAD.Rows[0][TABLES.Address.AddressGuid];
			drUpdAdd[TABLES.Address.AddressLine1] = "UnitTestChanged-" + System.DateTime.Now.ToString();
			drUpdAdd[TABLES.Address.AddressCity] = "West Chicago";
			drUpdAdd[TABLES.Address.AddressZipCode] = "60185";
			drUpdAdd[TABLES.Address.StateCode] = "IL";
			drUpdAdd[ARTIFICIAL.AddressRowVersion] = dtDataAD.Rows[0][TABLES.Address.RowVersion];
			dtUpdateAddress.Rows.Add(drUpdAdd);

			Common.Utility.AppendLastUpdateInformation(dtUpdateLS, Common.UpdateFunction.UnitTests);
			Common.Utility.AppendLastUpdateInformation(dtUpdateAddress, Common.UpdateFunction.UnitTests);

			bool retValue = DAL.LocalSupplier.UpdateLocalSupplier(dtUpdateLS, dtUpdateAddress);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(Common.RowVersionException))]
		public void UpdateLocalSupplier_Fail()
		{
			this.RefreshData = true;

			DataTable dtDataLS = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM LocalSupplier WHERE LocalSupplierGuid = '" + this._dtSupplier.Rows[0]["LocalSupplierGuid"].ToString() + "'");
			DataTable dtDataAD = UnitTestUtil.RunSQLGetDataTable("SELECT * FROM Address WHERE AddressGuid = '" + dtDataLS.Rows[0][TABLES.LocalSupplier.AddressGuid].ToString() + "'");

			DataTable dtUpdateLS = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(true);

			DataRow drUpdLS = dtUpdateLS.NewRow();
			drUpdLS[TABLES.LocalSupplier.LocalSupplierGuid] = dtDataLS.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid];
			drUpdLS[TABLES.LocalSupplier.FacilityName] = "UnitTestChanged-" + System.DateTime.Now.ToString();
			drUpdLS[TABLES.LocalSupplier.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.LocalEntry);
			drUpdLS[TABLES.LocalSupplier.RowVersion] = new byte[8];
			dtUpdateLS.Rows.Add(drUpdLS);

			DataTable dtUpdateAddress = DAL.Address.GetEmptyAddressTableSchema(true);

			DataRow drUpdAdd = dtUpdateAddress.NewRow();
			drUpdAdd[TABLES.Address.AddressGuid] = dtDataAD.Rows[0][TABLES.Address.AddressGuid];
			drUpdAdd[TABLES.Address.AddressLine1] = "UnitTestChanged-" + System.DateTime.Now.ToString();
			drUpdAdd[TABLES.Address.AddressCity] = "West Chicago";
			drUpdAdd[TABLES.Address.AddressZipCode] = "60185";
			drUpdAdd[TABLES.Address.StateCode] = "IL";
			drUpdAdd[ARTIFICIAL.AddressRowVersion] = new byte[8];
			dtUpdateAddress.Rows.Add(drUpdAdd);

			Common.Utility.AppendLastUpdateInformation(dtUpdateLS, Common.UpdateFunction.UnitTests);
			Common.Utility.AppendLastUpdateInformation(dtUpdateAddress, Common.UpdateFunction.UnitTests);

			DAL.LocalSupplier.UpdateLocalSupplier(dtUpdateLS, dtUpdateAddress);
		}

		
		[Test]
		public void GetEmptyLocalSupplierTableSchema_Pass()
		{
			DataTable dtTest = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(true);

			Assert.AreEqual(12, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.LocalSupplierGuid), TABLES.LocalSupplier.LocalSupplierGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Facility.FacilityName), TABLES.Facility.FacilityName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.LabelUnitIdTrimIndicator), TABLES.LocalSupplier.LabelUnitIdTrimIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityPrefix), TABLES.LocalSupplier.FacilityPrefix);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.PhoneNumber), TABLES.LocalSupplier.PhoneNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FaxNumber), TABLES.LocalSupplier.FaxNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.TestingIndicator), TABLES.LocalSupplier.TestingIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.CollectionIndicator), TABLES.LocalSupplier.CollectionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.RecordStatusCode), TABLES.LocalSupplier.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.IccbbaRegistrationNumber), TABLES.LocalSupplier.IccbbaRegistrationNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.LastUpdateFunctionId), TABLES.LocalSupplier.LastUpdateFunctionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.RowVersion), TABLES.LocalSupplier.RowVersion);

		}

		
		[Test]
		public void GetEmptyLocalSupplierTableSchema_Fail()
		{
			DataTable dtTest = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(false);

			Assert.AreEqual(15, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.LocalSupplierGuid), TABLES.LocalSupplier.LocalSupplierGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Facility.FacilityName), TABLES.Facility.FacilityName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.LabelUnitIdTrimIndicator), TABLES.LocalSupplier.LabelUnitIdTrimIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityPrefix), TABLES.LocalSupplier.FacilityPrefix);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.PhoneNumber), TABLES.LocalSupplier.PhoneNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FaxNumber), TABLES.LocalSupplier.FaxNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.TestingIndicator), TABLES.LocalSupplier.TestingIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.CollectionIndicator), TABLES.LocalSupplier.CollectionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.RecordStatusCode), TABLES.LocalSupplier.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.IccbbaRegistrationNumber), TABLES.LocalSupplier.IccbbaRegistrationNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.LastUpdateFunctionId), TABLES.LocalSupplier.LastUpdateFunctionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityGuid), TABLES.LocalSupplier.FacilityGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.AddressGuid), TABLES.LocalSupplier.AddressGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FdaRegistrationNumber), TABLES.LocalSupplier.FdaRegistrationNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.DivisionCode), TABLES.LocalSupplier.DivisionCode);
		}


		/// <summary>
		/// Reusable code to verify that the columns exist for spGetLocalSupplierLookupList
		/// </summary>
		/// <param name="dtTest"></param>
		private void VerifyLookupListResultDataTable( DataTable dtTest )
		{
			Assert.AreEqual(10,dtTest.Columns.Count,"Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.LocalSupplierGuid), TABLES.LocalSupplier.LocalSupplierGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Facility.FacilityGuid), TABLES.Facility.FacilityGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.IccbbaRegistrationNumber), TABLES.LocalSupplier.IccbbaRegistrationNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.PhoneNumber), TABLES.LocalSupplier.PhoneNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FaxNumber), TABLES.LocalSupplier.FaxNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.CollectionIndicator), TABLES.LocalSupplier.CollectionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.TestingIndicator), TABLES.LocalSupplier.TestingIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FdaRegistrationNumber), TABLES.LocalSupplier.FdaRegistrationNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.RecordStatusCode), TABLES.LocalSupplier.RecordStatusCode);
		}
		

	}
}
#endif
